<?php
session_start();

// Configuration
$enableAuth = true;
$storedHash = '$2a$12$wMwpjdVF1koCjqvlTKPlGeZ1aiJjnroho58ICR9FC18nWsUFQh3Lq'; // Replace with your bcrypt hash
$storagePath = './'; // Storage directory
$validTypes = ['php', 'pdf', 'jpg', 'jpeg', 'png', 'gif', 'zip']; // Allowed extensions
$maxFileSize = 5 * 1024 * 1024; // 5MB max file size
$sessionTimeout = 1800; // 30 minutes session timeout
$errorLogFile = 'error.log';

// Ensure storage directory exists
if (!is_dir($storagePath)) {
    mkdir($storagePath, 0755, true);
}

// Check session timeout
if ($enableAuth && isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > $sessionTimeout)) {
    session_unset();
    session_destroy();
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}
$_SESSION['last_activity'] = time();

// Check authentication
if ($enableAuth && !isset($_SESSION['is_logged_in'])) {
    if (isset($_POST['auth_key'])) {
        if (password_verify($_POST['auth_key'], $storedHash)) {
            $_SESSION['is_logged_in'] = true;
        } else {
            $authError = "Invalid password!";
            error_log("Authentication failed: Invalid password attempt at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
        }
    } else {
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Secure File Manager - Login</title>
            <script src="https://cdn.tailwindcss.com"></script>
        </head>
        <body class="bg-gray-100 flex items-center justify-center min-h-screen">
            <div class="bg-white p-8 rounded-lg shadow-lg w-full max-w-md">
                <h2 class="text-2xl font-bold mb-6 text-center">File Manager Login</h2>
                <?php if (isset($authError)) { echo "<p class='text-red-500 mb-4'>$authError</p>"; } ?>
                <form method="POST" class="space-y-4">
                    <div>
                        <label for="auth_key" class="block text-sm font-medium text-gray-700">Password</label>
                        <input type="password" id="auth_key" name="auth_key" required class="mt-1 w-full p-2 border rounded-md">
                    </div>
                    <button type="submit" class="w-full bg-blue-600 text-white p-2 rounded-md hover:bg-blue-700">Login</button>
                </form>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

// Handle file upload
if (isset($_FILES['uploaded_file'])) {
    $uploadedFile = $_FILES['uploaded_file'];
    $fileName = basename($uploadedFile['name']);
    $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    $targetPath = $storagePath . $fileName;

    if (!in_array($fileType, $validTypes)) {
        $uploadError = "Invalid file type!";
        error_log("Upload failed: Invalid file type ($fileType) at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
    } elseif ($uploadedFile['size'] > $maxFileSize) {
        $uploadError = "File size exceeds limit (5MB)!";
        error_log("Upload failed: File size too large at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
    } elseif (move_uploaded_file($uploadedFile['tmp_name'], $targetPath)) {
        $successMsg = "File uploaded successfully!";
    } else {
        $uploadError = "Failed to upload file!";
        error_log("Upload failed: Could not move file at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
    }
}

// Handle file deletion
if (isset($_GET['remove'])) {
    $fileToRemove = $storagePath . basename($_GET['remove']);
    if (file_exists($fileToRemove) && unlink($fileToRemove)) {
        $successMsg = "File deleted successfully!";
    } else {
        $uploadError = "Failed to delete file!";
        error_log("Deletion failed: Could not delete $fileToRemove at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
    }
}

// Handle file editing
if (isset($_POST['target_file']) && isset($_POST['new_content'])) {
    $fileToUpdate = $storagePath . basename($_POST['target_file']);
    if (file_exists($fileToUpdate) && is_writable($fileToUpdate)) {
        file_put_contents($fileToUpdate, $_POST['new_content']);
        $successMsg = "File updated successfully!";
    } else {
        $uploadError = "Failed to update file!";
        error_log("Update failed: Could not update $fileToUpdate at " . date('Y-m-d H:i:s'), 3, $errorLogFile);
    }
}

// Get and sort files
$fileList = array_diff(scandir($storagePath), ['.', '..']);
$sortBy = isset($_GET['sort']) && in_array($_GET['sort'], ['name', 'date']) ? $_GET['sort'] : 'name';
if ($sortBy === 'date') {
    usort($fileList, function($a, $b) use ($storagePath) {
        return filemtime($storagePath . $b) - filemtime($storagePath . $a);
    });
} else {
    natcasesort($fileList);
}

// Filter files by search
$searchQuery = isset($_GET['search']) ? strtolower(trim($_GET['search'])) : '';
if ($searchQuery) {
    $fileList = array_filter($fileList, function($file) use ($searchQuery) {
        return strpos(strtolower($file), $searchQuery) !== false;
    });
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Secure File Manager</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen p-6">
    <div class="max-w-4xl mx-auto bg-white rounded-lg shadow-lg p-6">
        <h2 class="text-3xl font-bold mb-6">Secure File Manager</h2>
        
        <?php if (isset($successMsg)) { echo "<p class='text-green-600 mb-4'>$successMsg</p>"; } ?>
        <?php if (isset($uploadError)) { echo "<p class='text-red-600 mb-4'>$uploadError</p>"; } ?>

        <!-- File Upload Form -->
        <div class="mb-8">
            <h3 class="text-xl font-semibold mb-4">Upload New File</h3>
            <form method="POST" enctype="multipart/form-data" class="space-y-4">
                <input type="file" name="uploaded_file" required class="w-full p-2 border rounded-md">
                <button type="submit" class="bg-blue-600 text-white p-2 rounded-md hover:bg-blue-700">Upload</button>
            </form>
        </div>

        <!-- Search and Sort -->
        <div class="mb-8 flex justify-between items-center">
            <form method="GET" class="flex-1 mr-4">
                <input type="text" name="search" value="<?php echo htmlspecialchars($searchQuery); ?>" placeholder="Search files..." class="w-full p-2 border rounded-md">
            </form>
            <div>
                <label for="sort" class="mr-2">Sort by:</label>
                <select id="sort" onchange="window.location.href='?sort='+this.value" class="p-2 border rounded-md">
                    <option value="name" <?php if ($sortBy === 'name') echo 'selected'; ?>>Name</option>
                    <option value="date" <?php if ($sortBy === 'date') echo 'selected'; ?>>Date</option>
                </select>
            </div>
        </div>

        <!-- File List -->
        <div class="mb-8">
            <h3 class="text-xl font-semibold mb-4">Files</h3>
            <?php if (empty($fileList)): ?>
                <p class="text-gray-600">No files found.</p>
            <?php else: ?>
                <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                    <?php foreach ($fileList as $file): ?>
                        <div class="border p-4 rounded-md flex items-center justify-between">
                            <div>
                                <?php
                                $fileType = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                                if (in_array($fileType, ['jpg', 'jpeg', 'png', 'gif'])) {
                                    echo "<img src='$storagePath$file' alt='$file' class='w-16 h-16 object-cover mr-4'>";
                                }
                                ?>
                                <a href="<?php echo $storagePath . $file; ?>" download class="text-blue-600 hover:underline"><?php echo $file; ?></a>
                            </div>
                            <div class="space-x-2">
                                <a href="?remove=<?php echo urlencode($file); ?>" onclick="return confirm('Are you sure you want to delete this file?');" class="text-red-600 hover:underline">Delete</a>
                                <a href="?modify=<?php echo urlencode($file); ?>" class="text-blue-600 hover:underline">Edit</a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- File Edit Form -->
        <?php if (isset($_GET['modify'])): ?>
            <?php
            $fileToModify = $storagePath . basename($_GET['modify']);
            $fileData = file_exists($fileToModify) ? file_get_contents($fileToModify) : '';
            ?>
            <div>
                <h3 class="text-xl font-semibold mb-4">Edit File: <?php echo basename($_GET['modify']); ?></h3>
                <form method="POST" class="space-y-4">
                    <textarea name="new_content" rows="10" required class="w-full p-2 border rounded-md"><?php echo htmlspecialchars($fileData); ?></textarea>
                    <input type="hidden" name="target_file" value="<?php echo basename($_GET['modify']); ?>">
                    <button type="submit" class="bg-blue-600 text-white p-2 rounded-md hover:bg-blue-700">Save Changes</button>
                </form>
            </div>
        <?php endif; ?>

        <!-- Logout -->
        <form method="POST" action="?signout" class="mt-8">
            <button type="submit" class="bg-red-600 text-white p-2 rounded-md hover:bg-red-700">Logout</button>
        </form>

        <?php
        // Handle logout
        if (isset($_GET['signout'])) {
            session_destroy();
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        }
        ?>
    </div>
</body>
</html>