<?php
// Enable error reporting for debugging (disable in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Configuration
$accessRestricted = true;
$storedPassHash = '$2a$12$wMwpjdVF1koCjqvlTKPlGeZ1aiJjnroho58ICR9FC18nWsUFQh3Lq'; // Replace with your bcrypt hash
$allowRemoteDownload = true;

// Security check for password protection
if ($accessRestricted) {
    if (!isset($_SESSION['auth_access']) || $_SESSION['auth_access'] !== true) {
        if (isset($_POST['auth_key'])) {
            if (password_verify($_POST['auth_key'], $storedPassHash)) {
                $_SESSION['auth_access'] = true;
            } else {
                die('
                <!DOCTYPE html>
                <html lang="en">
                <head>
                    <meta charset="UTF-8">
                    <meta name="viewport" content="width=device-width, initial-scale=1.0">
                    <title>Access Denied</title>
                    <script src="https://cdn.tailwindcss.com"></script>
                </head>
                <body class="bg-secure-gray min-h-screen flex items-center justify-center">
                    <div class="bg-white p-8 rounded-lg shadow-lg w-full max-w-md">
                        <h2 class="text-2xl font-bold mb-6 text-center text-secure-dark">Authentication Required</h2>
                        <form method="POST">
                            <input type="password" name="auth_key" class="w-full p-2 border rounded mb-4 focus:outline-none focus:ring-2 focus:ring-secure-blue" placeholder="Enter access key" required autofocus>
                            <button type="submit" class="w-full bg-secure-blue text-white p-2 rounded hover:bg-secure-blue-dark transition">Authenticate</button>
                        </form>
                    </div>
                </body>
                </html>
                ');
            }
        } else {
            echo '
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Access Denied</title>
                <script src="https://cdn.tailwindcss.com"></script>
            </head>
            <body class="bg-secure-gray min-h-screen flex items-center justify-center">
                <div class="bg-white p-8 rounded-lg shadow-lg w-full max-w-md">
                    <h2 class="text-2xl font-bold mb-6 text-center text-secure-dark">Authentication Required</h2>
                    <form method="POST">
                        <input type="password" name="auth_key" class="w-full p-2 border rounded mb-4 focus:outline-none focus:ring-2 focus:ring-secure-blue" placeholder="Enter access key" required autofocus>
                        <button type="submit" class="w-full bg-secure-blue text-white p-2 rounded hover:bg-secure-blue-dark transition">Authenticate</button>
                    </form>
                </div>
            </body>
            </html>
            ';
            exit;
        }
    }
}

// Handle file content retrieval for editing
if (isset($_GET['task']) && $_GET['task'] === 'fetch_file_data' && isset($_GET['target'])) {
    $securePath = realpath($_GET['base'] . '/' . $_GET['target']);
    if ($securePath && is_file($securePath) && is_readable($securePath)) {
        header('Content-Type: text/plain');
        echo file_get_contents($securePath);
        exit;
    } else {
        http_response_code(404);
        echo "Resource not found or not readable.";
        exit;
    }
}

// Handle current directory
$activePath = isset($_GET['base']) ? realpath($_GET['base']) : __DIR__;
if (!$activePath || !is_dir($activePath) || !is_readable($activePath)) {
    $activePath = __DIR__;
    $notification = "Invalid or inaccessible directory. Reverted to default directory.";
}

// File operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Create directory
    if (isset($_POST['new_directory']) && !empty($_POST['dir_name'])) {
        $dirName = cleanResourceName($_POST['dir_name']);
        $newDirPath = $activePath . '/' . $dirName;
        if (!is_dir($newDirPath)) {
            if (mkdir($newDirPath, 0755)) {
                $notification = "Directory created successfully.";
            } else {
                $notification = "Failed to create directory.";
            }
        } else {
            $notification = "Directory already exists.";
        }
    }

    // Create file
    if (isset($_POST['new_file']) && !empty($_POST['resource_name'])) {
        $fileName = cleanResourceName($_POST['resource_name']);
        $fileData = $_POST['resource_data'] ?? '';
        $newFilePath = $activePath . '/' . $fileName;
        if (!file_exists($newFilePath)) {
            if (file_put_contents($newFilePath, $fileData) !== false) {
                $notification = "File created successfully.";
            } else {
                $notification = "Failed to create file.";
            }
        } else {
            $notification = "File already exists.";
        }
    }

    // Upload file
    if (isset($_FILES['uploaded_resource']) && $_FILES['uploaded_resource']['error'] === UPLOAD_ERR_OK) {
        $uploadedResource = $_FILES['uploaded_resource'];
        $destinationPath = $activePath . '/' . cleanResourceName($uploadedResource['name']);
        if (!file_exists($destinationPath)) {
            if (move_uploaded_file($uploadedResource['tmp_name'], $destinationPath)) {
                $notification = "Resource uploaded successfully.";
            } else {
                $notification = "Error uploading resource.";
            }
        } else {
            $notification = "File already exists.";
        }
    }

    // Rename resource
    if (isset($_POST['rename_resource']) && !empty($_POST['current_name']) && !empty($_POST['updated_name'])) {
        $oldResource = cleanResourceName($_POST['current_name']);
        $newResource = cleanResourceName($_POST['updated_name']);
        $oldPath = $activePath . '/' . $oldResource;
        $newPath = $activePath . '/' . $newResource;
        if (file_exists($oldPath) && !file_exists($newPath)) {
            if (rename($oldPath, $newPath)) {
                $notification = "Resource renamed successfully.";
            } else {
                $notification = "Error renaming resource.";
            }
        } else {
            $notification = "Invalid source or destination name.";
        }
    }

    // Delete resource
    if (isset($_POST['remove_resource']) && !empty($_POST['resource_name'])) {
        $resourceName = cleanResourceName($_POST['resource_name']);
        $resourcePath = $activePath . '/' . $resourceName;
        if (file_exists($resourcePath)) {
            if (is_dir($resourcePath)) {
                if (count(scandir($resourcePath)) <= 2) { // Check if directory is empty
                    if (rmdir($resourcePath)) {
                        $notification = "Directory removed successfully.";
                    } else {
                        $notification = "Error removing directory.";
                    }
                } else {
                    $notification = "Directory is not empty.";
                }
            } elseif (is_file($resourcePath)) {
                if (unlink($resourcePath)) {
                    $notification = "File removed successfully.";
                } else {
                    $notification = "Error removing file.";
                }
            }
        } else {
            $notification = "Resource not found.";
        }
    }

    // Unzip archive
    if (isset($_POST['extract_archive']) && !empty($_POST['archive_name'])) {
        $archiveName = cleanResourceName($_POST['archive_name']);
        $archivePath = $activePath . '/' . $archiveName;
        if (file_exists($archivePath) && class_exists('ZipArchive')) {
            $archive = new ZipArchive;
            if ($archive->open($archivePath) === TRUE) {
                if ($archive->extractTo($activePath)) {
                    $archive->close();
                    $notification = "Archive extracted successfully.";
                } else {
                    $archive->close();
                    $notification = "Error extracting archive.";
                }
            } else {
                $notification = "Failed to open archive.";
            }
        } else {
            $notification = "Archive not found or ZipArchive not available.";
        }
    }

    // Fetch remote resource
    if ($allowRemoteDownload && isset($_POST['download_remote']) && !empty($_POST['remote_link'])) {
        $remoteLink = filter_var($_POST['remote_link'], FILTER_VALIDATE_URL);
        if ($remoteLink) {
            $resourceName = cleanResourceName(basename($remoteLink));
            $localResourcePath = $activePath . '/' . $resourceName;
            if (!file_exists($localResourcePath)) {
                $remoteContent = @file_get_contents($remoteLink);
                if ($remoteContent !== false && file_put_contents($localResourcePath, $remoteContent) !== false) {
                    $notification = "Resource downloaded successfully.";
                } else {
                    $notification = "Error downloading resource.";
                }
            } else {
                $notification = "File already exists.";
            }
        } else {
            $notification = "Invalid remote link.";
        }
    }

    // Edit file
    if (isset($_POST['update_file']) && !empty($_POST['resource_name'])) {
        $fileName = cleanResourceName($_POST['resource_name']);
        $fileData = $_POST['resource_data'] ?? '';
        $filePath = $activePath . '/' . $fileName;
        if (file_exists($filePath) && is_writable($filePath)) {
            if (file_put_contents($filePath, $fileData) !== false) {
                $notification = "File updated successfully.";
            } else {
                $notification = "Error updating file.";
            }
        } else {
            $notification = "File not found or not writable.";
        }
    }
}

// List directory contents
$resources = scandir($activePath);
$subDirectories = [];
$resourceFiles = [];

foreach ($resources as $resource) {
    if ($resource === '.' || $resource === '..') continue;
    $resourcePath = $activePath . '/' . $resource;
    if (is_dir($resourcePath)) {
        $subDirectories[] = $resource;
    } else {
        $resourceFiles[] = $resource;
    }
}
sort($subDirectories);
sort($resourceFiles);

// Helper functions
function cleanResourceName($name) {
    return preg_replace('/[^a-zA-Z0-9._-]/', '', trim($name));
}

function formatResourceSize($bytes) {
    if ($bytes >= 1073741824) return number_format($bytes / 1073741824, 2) . ' GB';
    if ($bytes >= 1048576) return number_format($bytes / 1048576, 2) . ' MB';
    if ($bytes >= 1024) return number_format($bytes / 1024, 2) . ' KB';
    return $bytes . ' bytes';
}

function generateNavPath($path) {
    $segments = array_filter(explode(DIRECTORY_SEPARATOR, $path));
    $navLinks = [];
    $pathBuilder = '';

    $navLinks[] = '<a href="?base=/" class="text-secure-blue hover:underline">Home</a>';
    foreach ($segments as $segment) {
        $pathBuilder .= DIRECTORY_SEPARATOR . $segment;
        if (realpath($pathBuilder)) {
            $navLinks[] = '<a href="?base=' . urlencode($pathBuilder) . '" class="text-secure-blue hover:underline">' . htmlspecialchars($segment) . '</a>';
        }
    }

    return implode(' / ', $navLinks);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Secure Resource Manager</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://kit.fontawesome.com/a076d05399.js" crossorigin="anonymous"></script>
    <style>
        .bg-secure-gray { background-color: #f0f4f8; }
        .bg-secure-blue { background-color: #3b82f6; }
        .bg-secure-blue-dark { background-color: #1d4ed8; }
        .text-secure-dark { color: #1f2937; }
        .modal-transition { transition: all 0.3s ease-in-out; }
        .hover-scale { transition: transform 0.2s; }
        .hover-scale:hover { transform: scale(1.05); }
    </style>
</head>
<body class="bg-secure-gray min-h-screen">
    <div class="container mx-auto p-6">
        <h1 class="text-4xl font-bold mb-8 text-secure-dark">Secure Resource Manager</h1>

        <!-- Navigation Path -->
        <nav class="mb-6 text-lg"><?php echo generateNavPath($activePath); ?></nav>

        <!-- Notification -->
        <?php if (isset($notification)): ?>
            <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded-lg" role="alert">
                <?php echo htmlspecialchars($notification); ?>
            </div>
        <?php endif; ?>

        <!-- Action Panels -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <!-- Create Directory -->
            <div class="bg-white p-6 rounded-lg shadow-lg hover-scale">
                <h3 class="text-xl font-semibold mb-4 text-secure-dark">New Directory</h3>
                <form method="post" class="flex space-x-3">
                    <input type="text" name="dir_name" class="flex-1 p-3 border rounded focus:outline-none focus:ring-2 focus:ring-secure-blue" placeholder="Directory name" required>
                    <button type="submit" name="new_directory" class="bg-secure-blue text-white p-3 rounded hover:bg-secure-blue-dark transition">Create</button>
                </form>
            </div>

            <!-- Upload Resource -->
            <div class="bg-white p-6 rounded-lg shadow-lg hover-scale">
                <h3 class="text-xl font-semibold mb-4 text-secure-dark">Upload Resource</h3>
                <form method="post" enctype="multipart/form-data" class="flex space-x-3">
                    <input type="file" name="uploaded_resource" class="flex-1 p-3 border rounded" required>
                    <button type="submit" class="bg-secure-blue text-white p-3 rounded hover:bg-secure-blue-dark transition">Upload</button>
                </form>
            </div>

            <!-- Download Remote Resource -->
            <?php if ($allowRemoteDownload): ?>
                <div class="bg-white p-6 rounded-lg shadow-lg hover-scale">
                    <h3 class="text-xl font-semibold mb-4 text-secure-dark">Download Remote Resource</h3>
                    <form method="post" class="flex space-x-3">
                        <input type="url" name="remote_link" class="flex-1 p-3 border rounded focus:outline-none focus:ring-2 focus:ring-secure-blue" placeholder="https://example.com/resource.zip" required>
                        <button type="submit" name="download_remote" class="bg-secure-blue text-white p-3 rounded hover:bg-secure-blue-dark transition">Download</button>
                    </form>
                </div>
            <?php endif; ?>
        </div>

        <!-- Resource Table -->
        <div class="bg-white rounded-lg shadow-lg overflow-x-auto">
            <table class="w-full">
                <thead class="bg-gray-100">
                    <tr>
                        <th class="p-4 text-left text-secure-dark">Name</th>
                        <th class="p-4 text-left text-secure-dark">Size</th>
                        <th class="p-4 text-left text-secure-dark">Writable</th>
                        <th class="p-4 text-left text-secure-dark">Last Modified</th>
                        <th class="p-4 text-left text-secure-dark">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <!-- Parent Directory -->
                    <?php if ($activePath !== DIRECTORY_SEPARATOR): ?>
                        <tr class="border-t hover:bg-gray-50">
                            <td class="p-4"><i class="fas fa-folder mr-2 text-secure-blue"></i><a href="?base=<?php echo urlencode(dirname($activePath)); ?>" class="text-secure-blue hover:underline">.. (Parent Directory)</a></td>
                            <td class="p-4">-</td>
                            <td class="p-4">-</td>
                            <td class="p-4">-</td>
                            <td class="p-4"></td>
                        </tr>
                    <?php endif; ?>

                    <!-- Subdirectories -->
                    <?php foreach ($subDirectories as $directory): ?>
                        <tr class="border-t hover:bg-gray-50">
                            <td class="p-4"><i class="fas fa-folder mr-2 text-secure-blue"></i><a href="?base=<?php echo urlencode($activePath . '/' . $directory); ?>" class="text-secure-blue hover:underline"><?php echo htmlspecialchars($directory); ?></a></td>
                            <td class="p-4">-</td>
                            <td class="p-4"><?php echo is_writable($activePath . '/' . $directory) ? 'Yes' : 'No'; ?></td>
                            <td class="p-4"><?php echo date("Y-m-d H:i:s", filemtime($activePath . '/' . $directory)); ?></td>
                            <td class="p-4 space-x-3">
                                <button onclick="openRenamePanel('<?php echo htmlspecialchars($directory); ?>')" class="bg-yellow-500 text-white px-3 py-1 rounded hover:bg-yellow-600 transition">Rename</button>
                                <button onclick="removeResource('<?php echo htmlspecialchars($directory); ?>')" class="bg-red-500 text-white px-3 py-1 rounded hover:bg-red-600 transition">Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>

                    <!-- Files -->
                    <?php foreach ($resourceFiles as $file): ?>
                        <tr class="border-t hover:bg-gray-50">
                            <td class="p-4"><i class="fas fa-file mr-2 text-secure-blue"></i><?php echo htmlspecialchars($file); ?></td>
                            <td class="p-4"><?php echo formatResourceSize(filesize($activePath . '/' . $file)); ?></td>
                            <td class="p-4"><?php echo is_writable($activePath . '/' . $file) ? 'Yes' : 'No'; ?></td>
                            <td class="p-4"><?php echo date("Y-m-d H:i:s", filemtime($activePath . '/' . $file)); ?></td>
                            <td class="p-4 space-x-3">
                                <button onclick="editResource('<?php echo htmlspecialchars($file); ?>')" class="bg-secure-blue text-white px-3 py-1 rounded hover:bg-secure-blue-dark transition">Edit</button>
                                <button onclick="openRenamePanel('<?php echo htmlspecialchars($file); ?>')" class="bg-yellow-500 text-white px-3 py-1 rounded hover:bg-yellow-600 transition">Rename</button>
                                <button onclick="removeResource('<?php echo htmlspecialchars($file); ?>')" class="bg-red-500 text-white px-3 py-1 rounded hover:bg-red-600 transition">Delete</button>
                                <?php if (pathinfo($file, PATHINFO_EXTENSION) === 'zip'): ?>
                                    <form method="post" class="inline">
                                        <input type="hidden" name="archive_name" value="<?php echo htmlspecialchars($file); ?>">
                                        <button type="submit" name="extract_archive" class="bg-green-500 text-white px-3 py-1 rounded hover:bg-green-600 transition"><i class="fas fa-file-archive mr-1"></i>Unzip</button>
                                    </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Edit Panel -->
    <div id="editPanel" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden items-center justify-center modal-transition">
        <div class="bg-white p-8 rounded-lg w-full max-w-3xl transform scale-95 modal-transition">
            <h2 class="text-2xl font-bold mb-6 text-secure-dark">Edit Resource</h2>
            <form id="editForm" method="post">
                <input type="hidden" id="editResourceName" name="resource_name">
                <textarea id="editResourceData" name="resource_data" class="w-full h-80 p-3 border rounded mb-6 focus:outline-none focus:ring-2 focus:ring-secure-blue"></textarea>
                <div class="flex justify-end space-x-3">
                    <button type="submit" name="update_file" class="bg-green-500 text-white px-5 py-2 rounded hover:bg-green-600 transition">Save</button>
                    <button type="button" onclick="closePanel('editPanel')" class="bg-gray-500 text-white px-5 py-2 rounded hover:bg-gray-600 transition">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Rename Panel -->
    <div id="renamePanel" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden items-center justify-center modal-transition">
        <div class="bg-white p-8 rounded-lg w-full max-w-md transform scale-95 modal-transition">
            <h2 class="text-2xl font-bold mb-6 text-secure-dark">Rename Resource</h2>
            <form id="renameForm" method="post">
                <input type="hidden" id="currentResourceName" name="current_name">
                <input type="text" id="updatedResourceName" name="updated_name" class="w-full p-3 border rounded mb-6 focus:outline-none focus:ring-2 focus:ring-secure-blue" required>
                <div class="flex justify-end space-x-3">
                    <button type="submit" name="rename_resource" class="bg-green-500 text-white px-5 py-2 rounded hover:bg-green-600 transition">Save</button>
                    <button type="button" onclick="closePanel('renamePanel')" class="bg-gray-500 text-white px-5 py-2 rounded hover:bg-gray-600 transition">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function editResource(fileName) {
            fetch('?base=<?php echo urlencode($activePath); ?>&task=fetch_file_data&target=' + encodeURIComponent(fileName))
                .then(response => {
                    if (!response.ok) throw new Error('File not found');
                    return response.text();
                })
                .then(content => {
                    document.getElementById('editResourceName').value = fileName;
                    document.getElementById('editResourceData').value = content;
                    openPanel('editPanel');
                })
                .catch(error => {
                    alert('Error loading file content: ' + error.message);
                });
        }

        function openRenamePanel(itemName) {
            document.getElementById('currentResourceName').value = itemName;
            document.getElementById('updatedResourceName').value = itemName;
            openPanel('renamePanel');
        }

        function removeResource(itemName) {
            if (confirm('Are you sure you want to delete this item?')) {
                const form = document.createElement('form');
                form.method = 'post';
                form.innerHTML = `
                    <input type="hidden" name="resource_name" value="${itemName}">
                    <input type="hidden" name="remove_resource" value="1">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        function openPanel(panelId) {
            const panel = document.getElementById(panelId);
            panel.classList.remove('hidden');
            panel.classList.add('flex');
            setTimeout(() => {
                panel.querySelector('.transform').classList.remove('scale-95');
            }, 10);
        }

        function closePanel(panelId) {
            const panel = document.getElementById(panelId);
            panel.querySelector('.transform').classList.add('scale-95');
            setTimeout(() => {
                panel.classList.add('hidden');
                panel.classList.remove('flex');
            }, 300);
        }
    </script>
</body>
</html>